/*
 * (C) 2021-2022 by sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
 * All Rights Reserved.
 *
 * Author: Neels Janosch Hofmeyr <nhofmeyr@sysmocom.de>
 *
 * SPDX-License-Identifier: GPL-2.0+
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <errno.h>
#include <string.h>

#include <osmocom/core/utils.h>

#include <osmocom/upf/upf.h>
#include <osmocom/upf/up_gtp_action.h>
#include <osmocom/upf/up_peer.h>
#include <osmocom/upf/up_session.h>

int up_gtp_action_cmp(const struct up_gtp_action *a, const struct up_gtp_action *b)
{
	int cmp;
	if (a == b)
		return 0;
	if (!a)
		return -1;
	if (!b)
		return 1;

#define CMP_MEMB(MEMB) OSMO_CMP(a->MEMB, b->MEMB)

	if ((cmp = CMP_MEMB(kind)))
		return cmp;

	switch (a->kind) {
	case UP_GTP_U_ENDECAPS:
		if ((cmp = CMP_MEMB(endecaps.local_teid)))
			return cmp;
		if ((cmp = CMP_MEMB(endecaps.remote_teid)))
			return cmp;
		cmp = osmo_sockaddr_cmp(&a->endecaps.gtp_remote_addr, &b->endecaps.gtp_remote_addr);
		if (cmp)
			return cmp;
		cmp = osmo_sockaddr_cmp(&a->endecaps.ue_addr, &b->endecaps.ue_addr);
		if (cmp)
			return cmp;
		break;

	case UP_GTP_U_TUNMAP:
		if ((cmp = CMP_MEMB(tunmap.access.local_teid)))
			return cmp;
		if ((cmp = CMP_MEMB(tunmap.access.remote_teid)))
			return cmp;
		if ((cmp = CMP_MEMB(tunmap.core.local_teid)))
			return cmp;
		if ((cmp = CMP_MEMB(tunmap.core.remote_teid)))
			return cmp;
		break;
	default:
		break;
	}
	return 0;
}

static int up_gtp_action_enable_disable(struct up_gtp_action *a, bool enable)
{
	struct upf_gtp_dev *gtp_dev;
	int rc;

	switch (a->kind) {
	case UP_GTP_U_ENDECAPS:
		if (g_upf->gtp.mockup) {
			LOG_UP_GTP_ACTION(a, LOGL_NOTICE, "gtp/mockup active, skipping GTP action %s\n",
					  enable ? "enable" : "disable");
			return 0;
		}

		/* use the first available GTP device.
		 * TODO: select by interface name?
		 */
		gtp_dev = upf_gtp_dev_first();
		if (!gtp_dev) {
			LOG_UP_GTP_ACTION(a, LOGL_ERROR, "No GTP device open, cannot %s\n", enable ? "enable" : "disable");
			return -EIO;
		}

		if (enable)
			rc = upf_gtp_dev_tunnel_add(gtp_dev, &a->endecaps);
		else
			rc = upf_gtp_dev_tunnel_del(gtp_dev, &a->endecaps);
		if (rc) {
			LOG_UP_GTP_ACTION(a, LOGL_ERROR, "Failed to %s GTP tunnel: %d %s\n",
					  enable ? "enable" : "disable", rc, strerror(-rc));
			return rc;
		}
		LOG_UP_GTP_ACTION(a, LOGL_NOTICE, "%s GTP tunnel\n", enable ? "Enabled" : "Disabled");
		return 0;

	case UP_GTP_U_TUNMAP:
		if (g_upf->nft.mockup) {
			LOG_UP_GTP_ACTION(a, LOGL_NOTICE, "nft/mockup active, skipping nftables ruleset %s\n",
					  enable ? "enable" : "disable");
			return 0;
		}

		if (enable && a->tunmap.id != 0) {
			LOG_UP_GTP_ACTION(a, LOGL_ERROR,
					  "Cannot enable: nft GTP tunnel mapping rule has been enabled before"
					  " as " NFT_CHAIN_NAME_PREFIX_TUNMAP "%u\n", a->tunmap.id);
			return -EALREADY;
		}
		if (!enable && a->tunmap.id == 0) {
			LOG_UP_GTP_ACTION(a, LOGL_ERROR,
					  "Cannot disable: nft GTP tunnel mapping rule has not been enabled"
					  " (no " NFT_CHAIN_NAME_PREFIX_TUNMAP " id)\n");
			return -ENOENT;
		}
		if (enable)
			rc = upf_nft_tunmap_create(&a->tunmap);
		else
			rc = upf_nft_tunmap_delete(&a->tunmap);
		if (rc) {
			LOG_UP_GTP_ACTION(a, LOGL_ERROR,
					  "Failed to %s nft GTP tunnel mapping " NFT_CHAIN_NAME_PREFIX_TUNMAP "%u:"
					  " %d %s\n", enable ? "enable" : "disable", a->tunmap.id, rc, strerror(-rc));
			return rc;
		}
		LOG_UP_GTP_ACTION(a, LOGL_NOTICE, "%s nft GTP tunnel mapping " NFT_CHAIN_NAME_PREFIX_TUNMAP "%u\n",
				  enable ? "Enabled" : "Disabled", a->tunmap.id);
		if (!enable)
			a->tunmap.id = 0;
		return 0;

	default:
		LOG_UP_GTP_ACTION(a, LOGL_ERROR, "Invalid action\n");
		return -ENOTSUP;
	}
}

int up_gtp_action_enable(struct up_gtp_action *a)
{
	return up_gtp_action_enable_disable(a, true);
}

int up_gtp_action_disable(struct up_gtp_action *a)
{
	return up_gtp_action_enable_disable(a, false);
}

int up_gtp_action_to_str_buf(char *buf, size_t buflen, const struct up_gtp_action *a)
{
	struct osmo_strbuf sb = { .buf = buf, .len = buflen };
	switch (a->kind) {
	case UP_GTP_U_ENDECAPS:
		OSMO_STRBUF_PRINTF(sb, "GTP:endecaps GTP-access:");
		OSMO_STRBUF_APPEND(sb, osmo_sockaddr_to_str_buf2, &a->endecaps.gtp_remote_addr);
		OSMO_STRBUF_PRINTF(sb, " TEID-r:0x%"PRIx32" TEID-l:0x%"PRIx32" IP-core:",
				   a->endecaps.remote_teid, a->endecaps.local_teid);
		OSMO_STRBUF_APPEND(sb, osmo_sockaddr_to_str_buf2, &a->endecaps.ue_addr);
		break;
	case UP_GTP_U_TUNMAP:
		OSMO_STRBUF_PRINTF(sb, "GTP:tunmap GTP-access:");
		OSMO_STRBUF_APPEND(sb, osmo_sockaddr_to_str_buf2, &a->tunmap.access.gtp_remote_addr);
		OSMO_STRBUF_PRINTF(sb, " TEID-access-r:0x%"PRIx32" TEID-access-l:0x%"PRIx32" GTP-core:",
				   a->tunmap.access.remote_teid, a->tunmap.access.local_teid);
		OSMO_STRBUF_APPEND(sb, osmo_sockaddr_to_str_buf2, &a->tunmap.core.gtp_remote_addr);
		OSMO_STRBUF_PRINTF(sb, " TEID-core-r:0x%"PRIx32" TEID-core-l:0x%"PRIx32,
				   a->tunmap.core.remote_teid, a->tunmap.core.local_teid);
		break;
	case UP_GTP_DROP:
		OSMO_STRBUF_PRINTF(sb, "GTP:drop");
		break;
	default:
		OSMO_STRBUF_PRINTF(sb, "GTP:?");
		break;
	}
	if (a->session)
		OSMO_STRBUF_PRINTF(sb, " PFCP-peer:%s SEID-l:0x%"PRIx64" PDR:%d,%d",
				   up_peer_remote_addr_str(a->session->up_peer),
				   a->session->up_seid, a->pdr_core, a->pdr_access);
	return sb.chars_needed;
}

char *up_gtp_action_to_str_c(void *ctx, const struct up_gtp_action *a)
{
	OSMO_NAME_C_IMPL(ctx, 128, "ERROR", up_gtp_action_to_str_buf, a)
}
